/*
 ---------------------------------------------------------------------------
 Copyright (c) 1998-2008, Brian Gladman, Worcester, UK. All rights reserved.

 LICENSE TERMS

 The redistribution and use of this software (with or without changes)
 is allowed without the payment of fees or royalties provided that:

  1. source code distributions include the above copyright notice, this
     list of conditions and the following disclaimer;

  2. binary distributions include the above copyright notice, this list
     of conditions and the following disclaimer in their documentation;

  3. the name of the copyright holder is not used to endorse products
     built using this software without specific written permission.

 DISCLAIMER

 This software is provided 'as is' with no explicit or implied warranties
 in respect of its properties, including, but not limited to, correctness
 and/or fitness for purpose.
 ---------------------------------------------------------------------------
 Issue Date: 20/12/2007
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <ctype.h>
#if defined( DUAL_CORE ) && defined( _WIN32 )
#include <windows.h>
#endif

#include "rdtsc.h"

#define cstr_cmp _strnicmp

enum des_type { KEY =  0, NCE, HDR, PTX, CTX, TAG,      /* sequence values      */
                VEC = 16, END, NUL, ERR, RPT, REM };    /* non-sequence values  */

/*  Input sequence designators are listed first with the most slowly
    varying values earlier in the list. Then output values are listed,
    followed by other values such as commands and aliases (e.g. IV is
    an alias for NCE).
*/
char *des_name[] =      { "KEY ", "NCE ", "HDR ", "PTX ", "CTX ", "TAG ", "IV  ", "VEC ", "END ", "RPT ", "REM " };
enum des_type des_m[] = {  KEY  ,  NCE  ,  HDR  ,  PTX  ,  CTX  ,  TAG  ,  NCE  ,  VEC  ,  END  ,  RPT  ,  REM   };

char*   in_dir  = "testvals/";
char*   out_dir = "outvals/";

typedef int  ret_type;
typedef ret_type t_init_and_key(const unsigned char*, unsigned long, void*);
typedef ret_type t_init_message(const unsigned char*, unsigned long, void*);
typedef ret_type t_auth_header (const unsigned char*, unsigned long, void*);
typedef ret_type t_auth_data   (const unsigned char*, unsigned long, void*);
typedef ret_type t_crypt_data  (unsigned char*, unsigned long, void*);
typedef ret_type t_compute_tag (unsigned char*, unsigned long, void*);
typedef ret_type t_encrypt     (unsigned char*, unsigned long, void*);
typedef ret_type t_decrypt     (unsigned char*, unsigned long, void*);
typedef ret_type t_encrypt_message(
            const unsigned char*, unsigned long,
            const unsigned char*, unsigned long,
            unsigned char*, unsigned long,
            unsigned char*, unsigned long,
            void*);
typedef ret_type t_decrypt_message(
            const unsigned char*, unsigned long,
            const unsigned char*, unsigned long,
            unsigned char*, unsigned long,
            unsigned char*, unsigned long,
            void*);
typedef ret_type t_end(void*);

typedef struct
{   char                *name;
    t_init_and_key      *init_and_key;
    t_init_message      *init_message;
    t_auth_header       *auth_header;
    t_auth_data         *auth_data;
    t_crypt_data        *crypt_data;
    t_compute_tag       *compute_tag;
    t_encrypt           *encrypt;
    t_decrypt           *decrypt;
    t_encrypt_message   *encrypt_message;
    t_decrypt_message   *decrypt_message;
    t_end               *end;
} mode_fns;

#undef mode

#include "ccm.h"
/* special init_message() call for CCM to replace   */
/* t_init_message below when CCM is being used      */
typedef ret_type s_init_message(const unsigned char*,
            unsigned long, length_t, length_t, unsigned long, void*);


#define mode(f) ccm_##f
void mode(functions)(mode_fns f[1])
{
    f->name = "CCM";
    f->init_and_key = mode(init_and_key);
    f->init_message = (t_init_message*)mode(init_message);
    f->auth_header = mode(auth_header);
    f->auth_data = mode(auth_data);
    f->crypt_data = mode(crypt_data);
    f->compute_tag = mode(compute_tag);
    f->encrypt  = mode(encrypt);
    f->decrypt  = mode(decrypt);
    f->encrypt_message  = mode(encrypt_message);
    f->decrypt_message  = mode(decrypt_message);
    f->end  = mode(end);
}
#undef mode

#include "cwc.h"
#define mode(f) cwc_##f
void mode(functions)(mode_fns f[1])
{
    f->name = "CWC";
    f->init_and_key = mode(init_and_key);
    f->init_message = mode(init_message);
    f->auth_header = mode(auth_header);
    f->auth_data = mode(auth_data);
    f->crypt_data = mode(crypt_data);
    f->compute_tag = mode(compute_tag);
    f->encrypt  = mode(encrypt);
    f->decrypt  = mode(decrypt);
    f->encrypt_message  = mode(encrypt_message);
    f->decrypt_message  = mode(decrypt_message);
    f->end  = mode(end);
}
#undef mode

#include "eax.h"
#define mode(f) eax_##f
void mode(functions)(mode_fns f[1])
{
    f->name = "EAX";
    f->init_and_key = mode(init_and_key);
    f->init_message = mode(init_message);
    f->auth_header = mode(auth_header);
    f->auth_data = mode(auth_data);
    f->crypt_data = mode(crypt_data);
    f->compute_tag = mode(compute_tag);
    f->encrypt  = mode(encrypt);
    f->decrypt  = mode(decrypt);
    f->encrypt_message  = mode(encrypt_message);
    f->decrypt_message  = mode(decrypt_message);
    f->end  = mode(end);
}
#undef mode

#include "gcm.h"
#define mode(f) gcm_##f
void mode(functions)(mode_fns f[1])
{
    f->name = "GCM";
    f->init_and_key = mode(init_and_key);
    f->init_message = mode(init_message);
    f->auth_header = mode(auth_header);
    f->auth_data = mode(auth_data);
    f->crypt_data = mode(crypt_data);
    f->compute_tag = mode(compute_tag);
    f->encrypt  = mode(encrypt);
    f->decrypt  = mode(decrypt);
    f->encrypt_message  = mode(encrypt_message);
    f->decrypt_message  = mode(decrypt_message);
    f->end  = mode(end);
}
#undef mode

#define BLOCK_SIZE AES_BLOCK_SIZE

/*  Find if a line starts with a recognised line designator and
    return its descriptor if it does or ERR if not.
*/
enum des_type find_des(char *s, char **cptr)
{   size_t i = strlen(s);

    while( i && (s[i - 1] < ' ' || s[i - 1] > '\x7e') )
        --i;
    s[i] = '\0';
    if( i == 0 )
        return NUL;
    for(i = 0; i < sizeof(des_name) / sizeof(des_name[0]); ++i)
        if(!strncmp(des_name[i], s, strlen(des_name[i])))
        {   
            *cptr = s + strlen(des_name[i]);
            return des_m[i];
        }
    return ERR;
}

int to_hex(char ch)
{
    return (isdigit(ch) ? 0 : 9) + (ch & 0x0f);
}

unsigned char hex_in(unsigned char *s)
{
    return 16 * to_hex(s[0]) + to_hex(s[1]);
}

unsigned int dec_in(char *s)
{   unsigned int n = 0;

    while( isdigit( *s ) )
        n = 10 * n + *s++ - '0';
    return n;
}

void do_test(mode_fns f[1], void* contx, int gen_flag)
{   char            tvi_name[64], tvo_name[64], line[80], *lp;
    FILE            *inf = 0, *outf = 0;
    unsigned char   key[32], iv[256], hdr[256], ptx[256], ctx[256],
                    tag[16], buf[256], tbuf[16], *p;
    int             key_len, iv_len, hdr_len, ptx_len, ctx_len, tag_len,
                    hdr_rpt, ptx_rpt, rpt_cnt, vec_no, *cnt, *rpt = &rpt_cnt, err, i;

    strcpy(tvi_name, in_dir);
    strcat(tvi_name, f->name);
    strcat(tvi_name, ".0");
    if( gen_flag )
    {
        strcpy(tvo_name, out_dir);
        strcat(tvo_name, f->name);
        strcat(tvo_name, ".0");
    }

    for( ; ; )
    {
        tvi_name[strlen(tvi_name) - 1]++;
        if( gen_flag )
        {
            tvo_name[strlen(tvo_name) - 1]++;
            lp = tvo_name - 1;
            while(*++lp)
                *lp = tolower(*lp);
        }
        if(!(inf = fopen(tvi_name, "r")))
            break;

        printf("\nUsing %s test vectors in \"%s\": ", f->name, tvi_name);
        do
        {
            fgets(line, 80, inf);
            if( cstr_cmp(line, "MDE ", 4) == 0 )
                break;
            if( cstr_cmp(line, "END ", 4) == 0 )
                break;
        }
        while
            ( !feof( inf) );

        if(feof(inf) || cstr_cmp(line, "END ", 4) == 0 )
            break;

        if(cstr_cmp(line + 4, f->name, 3) != 0)
        {
            printf("this file does not match %s", f->name);
            break;
        }

        if( gen_flag )
        {
            if(!(outf = fopen(tvo_name, "w")))
                break;
            fprintf(outf, "\nMDE %s", f->name);
        }

        for(err = 0; ; )
        {
            enum des_type   line_is;

            fgets(line, 80, inf);
            if(feof(inf))
                break;

            if(strlen(line) < 4)
            {
                key_len = iv_len = hdr_len = ptx_len =
                    ctx_len = tag_len = rpt_cnt = 0;
                hdr_rpt = ptx_rpt = 1;
                if( gen_flag )
                    fprintf(outf, "\n");
                continue;
            }

            switch( line_is = find_des(line, &lp) )
            {
            case VEC:   vec_no = dec_in(lp);
                        if( gen_flag )
                            fprintf(outf, "\nVEC %04i", vec_no);
                        continue;
            case RPT:   *rpt = dec_in(lp);
                        if( gen_flag )
                            fprintf(outf, "\nRPT %04i", *rpt);
                        continue;
            case NUL:
            case REM:   continue;

            case KEY:   p = key, cnt = &key_len, rpt = &rpt_cnt; break;
            case NCE:   p = iv,  cnt =  &iv_len, rpt = &rpt_cnt; break;
            case HDR:   p = hdr, cnt = &hdr_len, rpt = &hdr_rpt; break;
            case PTX:   p = ptx, cnt = &ptx_len, rpt = &ptx_rpt; break;
            case CTX:   p = ctx, cnt = &ctx_len, rpt = &rpt_cnt; break;
            case TAG:   p = tag, cnt = &tag_len, rpt = &rpt_cnt; break;

            case END:   break;
            case ERR:   printf("\nThe file \'%s\' contains an unrecognised line\n", tvi_name); 
                        break;
            }

            if(line_is == END)
                break;

            if( gen_flag )
            {
                if(line[strlen(line) - 1] == '\n' || line[strlen(line) - 1] == '\r')
                    line[strlen(line) - 1] = '\0';
                if(line_is != CTX && line_is != TAG)
                    fprintf(outf, "\n%s", line);
            }

            if(line_is == REM)
                continue;

            while(*lp != '\n' && *lp != '\0' && *(lp + 1) != '\n' && *(lp + 1) != '\0')
            {
                p[(*cnt)++] = hex_in(lp); lp += 2;
            }

            if(line_is != TAG)
                continue;

            f->init_and_key(key, key_len, contx);

            if(strcmp(f->name, "CCM") == 0)
            {
                int err = ((s_init_message*)f->init_message)(iv, iv_len,
                    hdr_len * hdr_rpt, ptx_len * ptx_rpt, tag_len, contx);
                if(err)
                {
                    if(err == CCM_AUTH_LENGTH_ERROR && hdr_len * hdr_rpt >= 65536 - 256)
                        continue;
                }
            }
            else
                f->init_message(iv, iv_len, contx);

            i = hdr_rpt;
            while(i--)
                f->auth_header(hdr, hdr_len, contx);
#if 1
            i = ptx_rpt;
            while(i--)
            {
                memcpy(buf, ptx, ptx_len);
                f->encrypt(buf, ptx_len, contx);
            }
            if(ptx_rpt == 1 && memcmp(ctx, buf, ctx_len))
                printf("\n\tencrypt ciphertext error on test number %i", vec_no), err++;
#else
            for (i=0; i < ptx_len; i += 16)
            {
                int len = min(16, ptx_len - i);
                memcpy(buf, ptx + i, len);
                f->encrypt(buf, len, contx);
                if(ptx_rpt == 1 && memcmp(ctx + i, buf, len))
                    printf("\n\tencrypt ciphertext error on test number %i", vec_no), err++;
            }
#endif
            f->compute_tag(tbuf, tag_len, contx);
            f->end(contx);

            if( gen_flag )
            {
                for(i = 0; i < ptx_len; ++i)
                {
                    if(i % 32 == 0) fprintf(outf, "\nCTX ");
                    fprintf(outf, "%02x", buf[i]);
                }

                for(i = 0; i < tag_len; ++i)
                {
                    if(i % 32 == 0) fprintf(outf, "\nTAG ");
                    fprintf(outf, "%02x", tbuf[i]);
                }
            }
            else if(memcmp(tag, tbuf, tag_len))
                printf("\n\tencrypt tag error on test number %i", vec_no), err++;

            f->init_and_key(key, key_len, contx);

            if(strcmp(f->name, "CCM") == 0)
                ((s_init_message*)f->init_message)(iv, iv_len,
                    hdr_len * hdr_rpt, ptx_len * ptx_rpt, tag_len, contx);
            else
                f->init_message(iv, iv_len, contx);

            i = hdr_rpt;
            while(i--)
                f->auth_header(hdr, hdr_len, contx);

            i = ptx_rpt;
            while(i--)
            {
                memcpy(buf, ctx, ptx_len);
                f->decrypt(buf, ptx_len, contx);
            }

            f->compute_tag(tbuf, tag_len, contx);
            f->end(contx);

            if(ptx_rpt == 1 && memcmp(ptx, buf, ptx_len))
                printf("\n\tdecrypt plaintext error on test number %i", vec_no), err++;
            if(memcmp(tag, tbuf, tag_len))
                printf("\n\tdecrypt tag error on test number %i", vec_no), err++;
        }

        if(!err)
            printf("matched");
        if(inf) fclose(inf);
        if( gen_flag && outf ) 
            fclose(outf);
    }
    return;
}

const int loops = 100; // number of timing loops

unsigned int rand32(void)
{   static unsigned int   r4,r_cnt = -1,w = 521288629,z = 362436069;

    z = 36969 * (z & 65535) + (z >> 16);
    w = 18000 * (w & 65535) + (w >> 16);

    r_cnt = 0; r4 = (z << 16) + w; return r4;
}

unsigned char rand8(void)
{   static unsigned int   r4,r_cnt = 4;

    if(r_cnt == 4)
    {
        r4 = rand32(); r_cnt = 0;
    }

    return (char)(r4 >> (8 * r_cnt++));
}

// fill a block with random charactrers

void block_rndfill(unsigned char l[], unsigned int len)
{   unsigned int  i;

    for(i = 0; i < len; ++i)

        l[i] = rand8();
}

double ccm_time(int key_len, int iv_len, int hdr_len, int txt_len)
{   int    i, c1 = INT_MAX, c2 = INT_MAX, cy1, cy2, err;
    unsigned volatile long long tval;
    unsigned char   t1[BLOCK_SIZE], t2[BLOCK_SIZE], t3[BLOCK_SIZE];
    ccm_ctx         ctx[1];

    unsigned char *kp = malloc(key_len);
    unsigned char *ip = malloc(iv_len);
    unsigned char *hp = malloc(hdr_len);
    unsigned char *tp = malloc(txt_len);
    unsigned char *bp1 = malloc(txt_len);
    unsigned char *bp2 = malloc(txt_len);
    unsigned char *bp3 = malloc(txt_len);
    block_rndfill(kp, key_len);
    block_rndfill(ip, iv_len);
    block_rndfill(hp, hdr_len);
    block_rndfill(tp, txt_len);

    ccm_init_and_key(kp, key_len, ctx);

    for(i = 0; i < loops; ++i)
    {
        memcpy(bp1, tp, txt_len);
        memcpy(bp2, tp, txt_len);
        memcpy(bp3, tp, txt_len);
        err = 0;

        tval = read_tsc();
        ccm_encrypt_message(ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx);
        cy1 = (int)(read_tsc() - tval);

        tval = read_tsc();
        ccm_encrypt_message(ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx);
        ccm_encrypt_message(ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx);
        cy2 = (int)(read_tsc() - tval);

        err |=  ccm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx) == EXIT_FAILURE
             || memcmp(bp1, tp, txt_len);

        err |=  ccm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx) == EXIT_FAILURE
             || memcmp(bp2, tp, txt_len);

        err |=  ccm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx) == EXIT_FAILURE
             || memcmp(bp3, tp, txt_len);

        if(err) printf("\n error");
        c1 = (unsigned int)(c1 > cy1 ? cy1 : c1);
        c2 = (unsigned int)(c2 > cy2 ? cy2 : c2);
    }

    ccm_end(ctx);
    free(kp); free(ip);
    free(hp); free(tp);
    free(bp1); free(bp2); free(bp3);

    return ((c2 - c1) + 0.5);
}

double cwc_time(int key_len, int iv_len, int hdr_len, int txt_len)
{   int    i, c1 = INT_MAX, c2 = INT_MAX, cy1, cy2, err;
    unsigned volatile long long tval;
    unsigned char   t1[BLOCK_SIZE], t2[BLOCK_SIZE], t3[BLOCK_SIZE];
    cwc_ctx         ctx[1];

    unsigned char *kp = malloc(key_len);
    unsigned char *ip = malloc(iv_len);
    unsigned char *hp = malloc(hdr_len);
    unsigned char *tp = malloc(txt_len);
    unsigned char *bp1 = malloc(txt_len);
    unsigned char *bp2 = malloc(txt_len);
    unsigned char *bp3 = malloc(txt_len);
    block_rndfill(kp, key_len);
    block_rndfill(ip, iv_len);
    block_rndfill(hp, hdr_len);
    block_rndfill(tp, txt_len);

    cwc_init_and_key(kp, key_len, ctx);

    for(i = 0; i < loops; ++i)
    {
        memcpy(bp1, tp, txt_len);
        memcpy(bp2, tp, txt_len);
        memcpy(bp3, tp, txt_len);
        err = 0;

        tval = read_tsc();
        cwc_encrypt_message(ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx);
        cy1 = (int)(read_tsc() - tval);

        tval = read_tsc();
        cwc_encrypt_message(ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx);
        cwc_encrypt_message(ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx);
        cy2 = (int)(read_tsc() - tval);

        err |=  cwc_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx) == EXIT_FAILURE
             || memcmp(bp1, tp, txt_len);

        err |=  cwc_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx) == EXIT_FAILURE
             || memcmp(bp2, tp, txt_len);

        err |=  cwc_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx) == EXIT_FAILURE
             || memcmp(bp3, tp, txt_len);

        if(err) printf("\n error");
        c1 = (unsigned int)(c1 > cy1 ? cy1 : c1);
        c2 = (unsigned int)(c2 > cy2 ? cy2 : c2);
    }

    cwc_end(ctx);
    free(kp); free(ip);
    free(hp); free(tp);
    free(bp1); free(bp2); free(bp3);

    return ((c2 - c1) + 0.5);
}

double eax_time(int key_len, int iv_len, int hdr_len, int txt_len)
{   int    i, c1 = INT_MAX, c2 = INT_MAX, cy1, cy2, err;
    unsigned volatile long long tval;
    unsigned char   t1[BLOCK_SIZE], t2[BLOCK_SIZE], t3[BLOCK_SIZE];
    eax_ctx         ctx[1];

    unsigned char *kp = malloc(key_len);
    unsigned char *ip = malloc(iv_len);
    unsigned char *hp = malloc(hdr_len);
    unsigned char *tp = malloc(txt_len);
    unsigned char *bp1 = malloc(txt_len);
    unsigned char *bp2 = malloc(txt_len);
    unsigned char *bp3 = malloc(txt_len);
    block_rndfill(kp, key_len);
    block_rndfill(ip, iv_len);
    block_rndfill(hp, hdr_len);
    block_rndfill(tp, txt_len);

    eax_init_and_key(kp, key_len, ctx);

    for(i = 0; i < loops; ++i)
    {
        memcpy(bp1, tp, txt_len);
        memcpy(bp2, tp, txt_len);
        memcpy(bp3, tp, txt_len);
        err = 0;

        tval = read_tsc();
        eax_encrypt_message(ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx);
        cy1 = (int)(read_tsc() - tval);

        tval = read_tsc();
        eax_encrypt_message(ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx);
        eax_encrypt_message(ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx);
        cy2 = (int)(read_tsc() - tval);

        err |=  eax_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx) == EXIT_FAILURE
             || memcmp(bp1, tp, txt_len);

        err |=  eax_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx) == EXIT_FAILURE
             || memcmp(bp2, tp, txt_len);

        err |=  eax_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx) == EXIT_FAILURE
             || memcmp(bp3, tp, txt_len);

        if(err) printf("\n error");
        c1 = (unsigned int)(c1 > cy1 ? cy1 : c1);
        c2 = (unsigned int)(c2 > cy2 ? cy2 : c2);
    }

    eax_end(ctx);
    free(kp); free(ip);
    free(hp); free(tp);
    free(bp1); free(bp2); free(bp3);

    return ((c2 - c1) + 0.5);
}

double gcm_time(int key_len, int iv_len, int hdr_len, int txt_len)
{   int    i, c1 = INT_MAX, c2 = INT_MAX, cy1, cy2, err;
    unsigned volatile long long tval;
    unsigned char   t1[BLOCK_SIZE], t2[BLOCK_SIZE], t3[BLOCK_SIZE];
    gcm_ctx         ctx[1];

    unsigned char *kp = malloc(key_len);
    unsigned char *ip = malloc(iv_len);
    unsigned char *hp = malloc(hdr_len);
    unsigned char *tp = malloc(txt_len);
    unsigned char *bp1 = malloc(txt_len);
    unsigned char *bp2 = malloc(txt_len);
    unsigned char *bp3 = malloc(txt_len);
    block_rndfill(kp, key_len);
    block_rndfill(ip, iv_len);
    block_rndfill(hp, hdr_len);
    block_rndfill(tp, txt_len);

    gcm_init_and_key(kp, key_len, ctx);

    for(i = 0; i < loops; ++i)
    {
        memcpy(bp1, tp, txt_len);
        memcpy(bp2, tp, txt_len);
        memcpy(bp3, tp, txt_len);
        err = 0;

        tval = read_tsc();
        gcm_encrypt_message(ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx);
        cy1 = (int)(read_tsc() - tval);

        tval = read_tsc();
        gcm_encrypt_message(ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx);
        gcm_encrypt_message(ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx);
        cy2 = (int)(read_tsc() - tval);

        err |=  gcm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp1, txt_len, t1, 16, ctx) == EXIT_FAILURE
             || memcmp(bp1, tp, txt_len);

        err |=  gcm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp2, txt_len, t2, 16, ctx) == EXIT_FAILURE
             || memcmp(bp2, tp, txt_len);

        err |=  gcm_decrypt_message
                  (ip, iv_len, hp, hdr_len, bp3, txt_len, t3, 16, ctx) == EXIT_FAILURE
             || memcmp(bp3, tp, txt_len);

        if(err) printf("\n error");
        c1 = (unsigned int)(c1 > cy1 ? cy1 : c1);
        c2 = (unsigned int)(c2 > cy2 ? cy2 : c2);
    }

    gcm_end(ctx);
    free(kp); free(ip);
    free(hp); free(tp);
    free(bp1); free(bp2); free(bp3);

    return ((c2 - c1) + 0.5);
}

static int tlen[12] = { 16, 20, 40, 44, 64, 128, 256, 552, 576, 1024, 1500, 8192 };

void ccm_tests(unsigned long key_len, unsigned long iv_len,
                                unsigned long hdr_len, double tval[], int flag)
{   ccm_ctx ctx[1];
    mode_fns f[1];
    int i;

    ccm_functions(f);
    do_test(f, ctx, flag);
    for(i = 0; i < 12; ++i)
        tval[i] = ccm_time(key_len, iv_len, hdr_len, tlen[i]);
}

void cwc_tests(unsigned long key_len, unsigned long iv_len,
                                unsigned long hdr_len, double tval[], int flag)
{   cwc_ctx ctx[1];
    mode_fns f[1];
    int i;

    cwc_functions(f);
    do_test(f, ctx, flag);
    for(i = 0; i < 12; ++i)
        tval[i] = cwc_time(key_len, iv_len, hdr_len, tlen[i]);
}

void eax_tests(unsigned long key_len, unsigned long iv_len,
                                unsigned long hdr_len, double tval[], int flag)
{   eax_ctx ctx[1];
    mode_fns f[1];
    int i;

    eax_functions(f);
    do_test(f, ctx, flag);
    for(i = 0; i < 12; ++i)
        tval[i] = eax_time(key_len, iv_len, hdr_len, tlen[i]);
}

void gcm_tests(unsigned long key_len, unsigned long iv_len,
                                unsigned long hdr_len, double tval[], int flag)
{   gcm_ctx ctx[1];
    mode_fns f[1];
    int i;

    gcm_functions(f);
    do_test(f, ctx, flag);
    for(i = 0; i < 12; ++i)
        tval[i] = gcm_time(key_len, iv_len, hdr_len, tlen[i]);
}

int main(void)
{
    double tval[4][13];
    unsigned long hdr_len = 0;
    int     i, flag = 1;

#if defined( DUAL_CORE ) && defined( _WIN32 )
    // we need to constrain the process to one core in order to
    // obtain meaningful timing data
    HANDLE ph;
    DWORD_PTR afp;
    DWORD_PTR afs;
    ph = GetCurrentProcess();
    if(GetProcessAffinityMask(ph, &afp, &afs))
    {
        afp &= (GetCurrentProcessorNumber() + 1);
        if(!SetProcessAffinityMask(ph, afp))
        {
            printf("Couldn't set Process Affinity Mask\n\n"); return -1;
        }
    }
    else
    {
        printf("Couldn't get Process Affinity Mask\n\n"); return -1;
    }
#endif

    ccm_tests(16, 12, hdr_len, tval[0], flag);
    cwc_tests(16, 12, hdr_len, tval[1], flag);
    eax_tests(16, 12, hdr_len, tval[2], flag);
    gcm_tests(16, 12, hdr_len, tval[3], flag);

    for(i = 0; i < 4; ++i)
    {   double av;
        av = 0.05 * tval[i][3] + 0.15 * tval[i][7]
                    + 0.2 * tval[i][8] + 0.6 * tval[i][10];
        av /= (0.05 * 44 + 0.15 * 552 + 0.2 * 576 + 0.6 * 1500 + hdr_len);
        tval[i][12] = av;
    }

    printf("\n\n  Length      CCM      CWC      EAX      GCM");
    for(i = 0; i < 12; ++i)
    {
        printf("\n%8i %8.2f %8.2f %8.2f %8.2f", tlen[i],
                    tval[0][i] / (tlen[i] + hdr_len),
                    tval[1][i] / (tlen[i] + hdr_len),
                    tval[2][i] / (tlen[i] + hdr_len),
                    tval[3][i] / (tlen[i] + hdr_len));
    }

    printf("\naverage: %8.2f %8.2f %8.2f %8.2f",
                    tval[0][12], tval[1][12], tval[2][12], tval[3][12]);

    printf("\n\n");
    return 0;
}



