/*
 ---------------------------------------------------------------------------
 Copyright (c) 1998-2008, Brian Gladman, Worcester, UK. All rights reserved.

 LICENSE TERMS

 The redistribution and use of this software (with or without changes)
 is allowed without the payment of fees or royalties provided that:

  1. source code distributions include the above copyright notice, this
     list of conditions and the following disclaimer;

  2. binary distributions include the above copyright notice, this list
     of conditions and the following disclaimer in their documentation;

  3. the name of the copyright holder is not used to endorse products
     built using this software without specific written permission.

 DISCLAIMER

 This software is provided 'as is' with no explicit or implied warranties
 in respect of its properties, including, but not limited to, correctness
 and/or fitness for purpose.
 ---------------------------------------------------------------------------
 Issue Date: 26/08/2003
*/

#include <stdio.h>

#include <memory.h>

#include "omac-aes.h"

/* These values are used to detect long word alignment in order */
/* to speed up some CCM buffer operations. This facility may    */
/* need to be disabled (by setting A_PWR to 0) on some machines */

#include <limits.h>
#if ULONG_MAX == 0xffffffff
#define A_PWR   2
#elif ULONG_MAX == 0xffffffffffffffff
#define A_PWR   3
#else
#define A_PWR   0
#endif

#define A_SIZE      (1 << A_PWR)
#define A_MASK      (A_SIZE - 1)
#define lp(x)       ((unsigned long*)(x))
#define aligned(x)  (!(((unsigned long)(x)) & A_MASK)) 

#define BLOCK_SIZE  AES_BLOCK_SIZE

char    *msg[] =
{
    "Key In:   ",
    "Data:     ",
    "Pre-Enc-1 ",
    "Post-Enc-1",
    "Pre-Enc-2 ",
    "Post-Enc-2",
    "Pre-Enc-3 ",
    "Post-Enc-3",
    "L         ",
    "Lu        ",
    "Lu^2      ",
    "Lu^-1     ",
    "Pre-Xor   ",
    "Post-Xor  ",
    "Tag       "
};

void out_block(int msg_no, const void *bp, int len)
{   int i;

    printf("\n%s ", msg[msg_no]);

    for(i = 0; i < len; ++i)
    {   if(i && i % 8 == 0)
            printf("\n           ");
        printf("0x%02X ", ((unsigned char*)bp)[i]);
    }
}

void omac_init(const unsigned char key[], unsigned long key_len, omac_ctx ctx[1])
{
    memset(ctx, 0, sizeof(omac_ctx));
    aes_encrypt_key(key, key_len, ctx->aes);
    out_block(0, key, key_len);
}

void omac_data(unsigned char buf[], unsigned long len, omac_ctx ctx[1])
{   unsigned int cnt = 0, b_pos = ctx->txt_cnt;

    out_block(1, buf, len);
    if(aligned(ctx->txt_cbc))
    {
        while((b_pos & A_MASK) && cnt < len)
            ctx->txt_cbc[b_pos++] ^= buf[cnt++];

        if(!(b_pos & A_MASK) && !((buf + cnt - ctx->txt_cbc) & A_MASK))
        {   unsigned long *bp = lp(buf + cnt);

            while(cnt + BLOCK_SIZE <= len)
            {
                switch(b_pos)
                {
                case BLOCK_SIZE:
                         out_block(2, ctx->txt_cbc, BLOCK_SIZE);
                         aes_encrypt(ctx->txt_cbc, ctx->txt_cbc, ctx->aes); 
                         out_block(3, ctx->txt_cbc, BLOCK_SIZE);
                         b_pos = 0;
#if A_PWR == 2
                case  0: lp(ctx->txt_cbc)[0] ^= *bp++;
                case  4: lp(ctx->txt_cbc)[1] ^= *bp++;
                case  8: lp(ctx->txt_cbc)[2] ^= *bp++;
                case 12: lp(ctx->txt_cbc)[3] ^= *bp++;
#else
                case  0: lp(ctx->txt_cbc)[0] ^= *bp++;
                case  8: lp(ctx->txt_cbc)[1] ^= *bp++;
#endif
                }
                cnt += BLOCK_SIZE - b_pos; b_pos = BLOCK_SIZE;
            }
        }
    }

    while(cnt + BLOCK_SIZE <= len)
    {   unsigned char *bp = buf + cnt;

        switch(b_pos)
        {
        case BLOCK_SIZE:
                 out_block(4, ctx->txt_cbc, BLOCK_SIZE);
                 aes_encrypt(ctx->txt_cbc, ctx->txt_cbc, ctx->aes); 
                 out_block(5, ctx->txt_cbc, BLOCK_SIZE);
                 b_pos = 0;
        case  0: ctx->txt_cbc[ 0] ^= *bp++;
        case  1: ctx->txt_cbc[ 1] ^= *bp++;
        case  2: ctx->txt_cbc[ 2] ^= *bp++;
        case  3: ctx->txt_cbc[ 3] ^= *bp++;
        case  4: ctx->txt_cbc[ 4] ^= *bp++;
        case  5: ctx->txt_cbc[ 5] ^= *bp++;
        case  6: ctx->txt_cbc[ 6] ^= *bp++;
        case  7: ctx->txt_cbc[ 7] ^= *bp++;
        case  8: ctx->txt_cbc[ 8] ^= *bp++;
        case  9: ctx->txt_cbc[ 9] ^= *bp++;
        case 10: ctx->txt_cbc[10] ^= *bp++;
        case 11: ctx->txt_cbc[11] ^= *bp++;
        case 12: ctx->txt_cbc[12] ^= *bp++;
        case 13: ctx->txt_cbc[13] ^= *bp++;
        case 14: ctx->txt_cbc[14] ^= *bp++;
        case 15: ctx->txt_cbc[15] ^= *bp++;
        }
        cnt += BLOCK_SIZE - b_pos; b_pos = BLOCK_SIZE;
    }

    while(cnt < len)
    {
        if(b_pos == BLOCK_SIZE)
        {
            out_block(6, ctx->txt_cbc, BLOCK_SIZE);
            aes_encrypt(ctx->txt_cbc, ctx->txt_cbc, ctx->aes);
            out_block(7, ctx->txt_cbc, BLOCK_SIZE);
            b_pos = 0;
        }
        ctx->txt_cbc[b_pos++] ^= buf[cnt++];
    }

    ctx->txt_cnt = b_pos;
}

static const unsigned char c_xor[4] = { 0x00, 0x87, 0x0e, 0x89 };

void gf_mulx(unsigned char pad[BLOCK_SIZE])
{   int i, t = pad[0] >> 7;

    for(i = 0; i < BLOCK_SIZE - 1; ++i)
        pad[i] = (pad[i] << 1) | (pad[i + 1] >> 7);
    pad[BLOCK_SIZE - 1] = (pad[BLOCK_SIZE - 1] << 1) ^ c_xor[t];
}

#if OMAC_VERSION == 1

void gf_mulx2(unsigned char pad[BLOCK_SIZE])
{   int i, t = pad[0] >> 6;   
    
    for(i = 0; i < BLOCK_SIZE - 1; ++i)
        pad[i] = (pad[i] << 2) | (pad[i + 1] >> 6);
    pad[BLOCK_SIZE - 2] ^= (t >> 1);
    pad[BLOCK_SIZE - 1] = (pad[BLOCK_SIZE - 1] << 2) ^ c_xor[t];
}

#else

void gf_divx(unsigned char pad[BLOCK_SIZE])
{   int i, t;

    for(i = BLOCK_SIZE - 1, t = pad[BLOCK_SIZE - 1]; i > 0; --i)
        pad[i] = (pad[i] >> 1) | (pad[i - 1] << 7);
    pad[0] = (pad[0] >> 1) ^ ((t & 1) ? 0x80 : 0);
    pad[BLOCK_SIZE - 1] ^= ((t & 1) ? 0x43 : 0);
}

#endif

#if 0   /* little endian counters   */

void gf_mulx(unsigned char pad[BLOCK_SIZE])
{   int i, t;

    for(i = BLOCK_SIZE - 1, t = pad[BLOCK_SIZE - 1]; i > 0; --i)
        pad[i] = (pad[i] << 1) | (pad[i - 1] >> 7);
    pad[0] = (pad[0] << 1) ^ ((t & 0x80) ? 0x87 : 0);
}

void gf_divx(unsigned char pad[BLOCK_SIZE])
{   int i, t;

    for(i = 0, t = pad[0]; i < BLOCK_SIZE - 1; ++i)
        pad[i] = (pad[i] >> 1) | (pad[i + 1] << 7);
    pad[BLOCK_SIZE - 1] = (pad[BLOCK_SIZE - 1] >> 1) ^ ((t & 1) ? 0x80 : 0);
    pad[0] ^= ((t & 1) ? 0x43 : 0);
}

#endif

void omac_end(unsigned char auth_tag[], omac_ctx ctx[1])
{   unsigned char   pad[BLOCK_SIZE];
    int             i;

    memset(pad, 0, sizeof(pad));
    aes_encrypt(pad, pad, ctx->aes);
    out_block(8, pad, BLOCK_SIZE);

    if(ctx->txt_cnt < BLOCK_SIZE)
    {
        ctx->txt_cbc[ctx->txt_cnt] ^= 0x80;
#if OMAC_VERSION == 1
        gf_mulx2(pad);
        out_block(10, pad, BLOCK_SIZE);
#else
        gf_divx(pad);
        out_block(11, pad, BLOCK_SIZE);
#endif
    }
    else
    {
        gf_mulx(pad);
        out_block(9, pad, BLOCK_SIZE);
    }

    out_block(12, ctx->txt_cbc, BLOCK_SIZE);
    if(aligned(pad) && aligned(ctx->txt_cbc))
    {
        lp(pad)[0] ^= lp(ctx->txt_cbc)[0];
        lp(pad)[1] ^= lp(ctx->txt_cbc)[1];
#if A_PWR == 2
        lp(pad)[2] ^= lp(ctx->txt_cbc)[2];
        lp(pad)[3] ^= lp(ctx->txt_cbc)[3];
#endif
    }
    else
    {
        pad[ 0] ^= ctx->txt_cbc[ 0]; pad[ 1] ^= ctx->txt_cbc[ 1];
        pad[ 2] ^= ctx->txt_cbc[ 2]; pad[ 3] ^= ctx->txt_cbc[ 3];
        pad[ 4] ^= ctx->txt_cbc[ 4]; pad[ 5] ^= ctx->txt_cbc[ 5];
        pad[ 6] ^= ctx->txt_cbc[ 6]; pad[ 7] ^= ctx->txt_cbc[ 7];
        pad[ 8] ^= ctx->txt_cbc[ 8]; pad[ 9] ^= ctx->txt_cbc[ 9];
        pad[10] ^= ctx->txt_cbc[10]; pad[11] ^= ctx->txt_cbc[11];
        pad[12] ^= ctx->txt_cbc[12]; pad[13] ^= ctx->txt_cbc[13];
        pad[14] ^= ctx->txt_cbc[14]; pad[15] ^= ctx->txt_cbc[15];
    }

    out_block(13, pad, BLOCK_SIZE);
    aes_encrypt(pad, pad, ctx->aes);
    out_block(14, pad, BLOCK_SIZE);

    for(i = 0; i < BLOCK_SIZE; ++i)
        auth_tag[i] = pad[i];
}

